#pragma rtGlobals=1		// Use modern global access method.

Menu "Macros"
	SubMenu "Style"
		"Colorize traces", colorize_traces("")
		"Style small graph", style_small_graph()
		help = {"Adjust graph styling for small (paper) graphs."}
	End
	SubMenu "Plot"
		"Temperature data", PlotTemperatureData()
		help = {"Plots temperature data recorded using LabView."}
	End
	SubMenu "Waves"
		"Create sum wave", SumWaves()
		help = {"Sums up the currently selected waves point by point."}
		"Create rescaled waves", RescaleWaves()
		help = {"Rescales waves using an affine transformation."}
		"Create rescaled waves from graph", RescaleWavesFromGraph()
		help = {"Rescales waves using an affine transformation."}
		"Extract values at x=x0", GetValuesAtX()
		help = {"Extract data from each trace in the current plot at a given x."}
		"Merge trace data", MergeTraceData()
		help = {"Merges data from the current plot, grouped by trace style"}
	End
End

Static Function/S build_sorted_list(list, sep, n)
	String list, sep
	Variable n

	Variable k
	String sorted_list
	Wave tmp_sort_idx = tmp_sort_idx

	sorted_list = StringFromList(tmp_sort_idx[0], list, sep)
	for (k = 1; k < n; k += 1)
		sorted_list = sorted_list + sep + StringFromList(tmp_sort_idx[k], list, sep)
	endfor
	KillWaves tmp_sort_idx
	return sorted_list
End

Function/S sort_list_1(list, format_str, sep)
	String list, format_str, sep

	Variable k, n, a1
	String item
	n = ItemsInList(list, sep)
	if (n < 2)
		return list
	endif

	Make /I /N=(n) tmp_sort_idx, tmp_sort_keys, tmp_sort_length
	tmp_sort_idx = p
	for (k = 0; k < n; k += 1)
		item = StringFromList(k, list, sep)
		sscanf item, format_str, a1
		tmp_sort_keys[k] = a1
		tmp_sort_length[k] = strlen(item)
	endfor

	Sort {tmp_sort_keys, tmp_sort_length}, tmp_sort_idx
	KillWaves tmp_sort_keys, tmp_sort_length
	return build_sorted_list(list, sep, n)
End

Function/S sort_list_2(list, format_str, sep)
	String list, format_str, sep

	Variable k, n, a1, a2
	String trace, sorted_list
	n = ItemsInList(list, sep)
	if (n < 2)
		return list
	endif

	Make /I /N=(n) tmp_sort_idx, tmp_sort_keys1, tmp_sort_keys2, tmp_sort_length
	tmp_sort_idx = p
	for (k = 0; k < n; k += 1)
		trace = StringFromList(k, list, sep)
		sscanf trace, format_str, a1, a2
		tmp_sort_keys1[k] = a1
		tmp_sort_keys2[k] = a2
		tmp_sort_length[k] = strlen(trace)
	endfor

	Sort {tmp_sort_keys1, tmp_sort_keys2, tmp_sort_length}, tmp_sort_idx
	KillWaves tmp_sort_keys1, tmp_sort_keys2, tmp_sort_length
	return build_sorted_list(list, sep, n)
End

Function replace_in_text_wave(w, search_str, replace_str)
	Wave/T w
	String search_str, replace_str

	Variable k, n
	n = numpnts(w)
	for (k = 0; k < n; k += 1)
		w[k] = ReplaceString(search_str, w[k], replace_str)
	endfor
End

Function load_cycler_data(path, [pattern])
	// Examples:
	//   load_cycler_data(":filtered_data")
	//   load_cycler_data(":filtered_data", pattern="*.txt")
	String path, pattern

	Variable k
	String filename, dirname, data_folder
	if (ParamIsDefault(pattern))
		pattern = ""
	endif

	data_folder = GetDataFolder(1)
	if (cmpstr(path[0], ":") == 0)
		// relative path
		PathInfo home
		NewPath/Q tmp_cycler_data (S_path + path[1,strlen(path)-1])
	else
		NewPath/Q tmp_cycler_data path
	endif
	PathInfo tmp_cycler_data
	if (V_flag == 0)
		return 1
	endif

	for (k = 0;; k += 1)
		filename = TextFile(tmp_cycler_data, k)
		if (strlen(filename) == 0)
			break
		endif
		if (strlen(pattern) > 0 && !stringmatch(filename, pattern))
			continue
		endif
		dirname = RemoveEnding(filename, ".txt")
		NewDataFolder/S $dirname
		LoadWave/J/D/W/A/K=0/P=tmp_cycler_data filename
		SetDataFolder data_folder
	endfor
	KillPath tmp_cycler_data
	return 0
End

Function load_colors(N, [colors])
	Variable N
	String colors

	if (ParamIsDefault(colors) || strlen(colors) == 0)
		ColorTab2Wave Rainbow
	else
		ColorTab2Wave $colors
	endif
	Wave M_colors = M_colors
	SetScale/I x, 0, N, M_colors
	Redimension /D M_colors
	M_colors *= 0.7
	Redimension /W/U M_colors
End

Function style_small_graph()
	ModifyGraph width=130,height=130,margin(right)=12
	ModifyGraph gfSize=12,axOffset(left)=-1.5,axOffset(bottom)=-0.3
	ModifyGraph nticks(left)=3,sep=10
End

Function colorize_traces(traces, [format_str, colors, flat, pattern])
	// Usage example: colorize_traces("", format_str="%*[^0123456789]%d_%*s")
	String traces, format_str, colors, pattern
	Variable flat

	if (ParamIsDefault(format_str))
		format_str = ""
	endif
	if (ParamIsDefault(flat))
		flat = 0
	endif
	if (ParamIsDefault(pattern))
		pattern = ""
	endif
//	if (ParamIsDefault(select_traces))
//		select_traces = ""
//	endif
	if (strlen(traces) == 0)
		traces = TraceNameList("", ";", 1)
//		if (strlen(select_traces) > 0)
//			traces = ListMatch(traces, select_traces, ";")
//		endif
		if (strlen(format_str) > 0)
			traces = sort_list_1(traces, format_str, ";")
		endif
	endif

	String trace
	Variable a1, k, kc, n, idx
	if (strlen(pattern) > 0)
		for (k = 0;;)
			trace = StringFromList(k, traces)
			if (strlen(trace) == 0)
				break
			endif
			if (stringmatch(trace, pattern))
				k += 1
			else
				traces = RemoveListItem(k, traces)
			endif
		endfor
	endif
	for (k = 0, n = 0;; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		if (flat || !stringmatch(trace, "*#*"))
			n += 1
		endif
	endfor

	Make /N=(n)/I tmp_color_map, tmp_linestyle_map
	Wave /I tmp_color_map = tmp_color_map
	Wave /I tmp_linestyle_map = tmp_linestyle_map
	if (strlen(format_str) > 0)
		Make /N=(n) tmp_unique_values
		Wave tmp_unique_values = tmp_unique_values
		tmp_unique_values = NaN
		for (k = 0, kc = 0;; k += 1)
			trace = StringFromList(k, traces)
			if (strlen(trace) == 0)
				break
			endif
			if (stringmatch(trace, "*#*"))
				continue
			endif
			sscanf trace, format_str, a1
			FindValue /V=(a1) tmp_unique_values
			if (V_value < 0)
				tmp_color_map[k] = kc
				tmp_linestyle_map[k] = 0
				tmp_unique_values[kc] = a1
				kc += 1
			else
				tmp_color_map[k] = V_value
				tmp_linestyle_map[k] = 8
				n -= 1
			endif
		endfor
		KillWaves tmp_unique_values
	else
		tmp_color_map = p
		tmp_linestyle_map = 0
	endif

	if (ParamIsDefault(colors))
		load_colors(n)
	else
		load_colors(n, colors=colors)
	endif

	Wave M_colors = M_colors
	for (k = 0, kc = -1;; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		if (k == 0 || flat || !stringmatch(trace, "*#*"))
			kc += 1
		endif
		idx = tmp_color_map[kc]
		ModifyGraph rgb($trace)=(M_colors(idx)[0],M_colors(idx)[1],M_colors(idx)[2])
		if (flat || !stringmatch(trace, "*#*"))
			ModifyGraph lstyle($trace)=tmp_linestyle_map[kc]
		else
			ModifyGraph lstyle($trace)=8
		endif
	endfor
	KillWaves M_colors, tmp_color_map, tmp_linestyle_map
End

Macro PlotTemperatureData(pattern, x_suffix)
	String pattern = "*_Temp", x_suffix = "_time"
	Prompt pattern, "pattern of the temperature waves' names"
	Prompt x_suffix, "suffix of time waves associated with each temperature wave"

	plot_temperature_data(pattern=pattern, x_suffix=x_suffix)
End

Function plot_temperature_data([pattern, x_suffix])
	String pattern, x_suffix
	
	Variable k, n, L
	String wname, wlist
	
	if (ParamIsDefault(pattern))
		pattern = "*_Temp"
	endif
	if (ParamIsDefault(x_suffix))
		x_suffix = "_time"
	endif
	n = strsearch(pattern, "*", 0)
	if (n < 0)
		String message
		sprintf message, "Pattern \"%s\" does not containg \"*\" character." , pattern
		Abort message
	else
		n = strlen(pattern) - n;
	endif
	
	Display as GetDataFolder(0)
	wlist = WaveList(pattern, ";", "")
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		L = strlen(wname)
		AppendToGraph $wname vs $(wname[0,L - n] + x_suffix)
	endfor
	ModifyGraph minor=1
	Label left "Temperature [C]"
	Label bottom "Time [\\u#2s]"
End

Function diff_waves(xwave, [output_folder])
	Wave xwave
	String output_folder

	Variable k
	String wlist, wname
	if (ParamIsDefault(output_folder))
		output_folder="diff"
	endif
	NewDataFolder/O $output_folder
	wlist = WaveList("*", ";", "")
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		if (WaveExists(xwave))
			Differentiate $wname/X=xwave/D=:$(output_folder):$wname
		else
			Differentiate $wname/D=:$(output_folder):$wname
		endif
	endfor
End

Function/S get_selected_waves()
	String selection, item
	Variable k

	for (selection = "", k = 0;; k += 1)
		item = GetBrowserSelection(k)
		if (strlen(item) == 0)
			break
		endif
		if (WaveExists($item))
			selection = AddListItem(GetWavesDataFolder($item, 4), selection)
		endif
	endfor
	return selection
End

Function SumWaves()
	// Sums up waves selected in the data browser.

	String target, selection, text
	target = "sum_wave"
	selection = get_selected_waves()
	if (strlen(selection) == 0)
		Abort "No waves selected." 
	endif

	Prompt target, "Target wave name:"
	sprintf text, "Sum waves %s", ReplaceString(";", selection, ", ")
	DoPrompt text, target
	if (V_flag != 0)
		return NaN
	endif
	printf "sum_waves(\"%s\", \"%s\")\r", selection, target
	return sum_waves(selection, target)
End

Function sum_waves(wlist, target, [min_value, pattern])
	// Creates a sum wave where w_sum[p] = w1[p] + w2[p] + ...
	String wlist, target, pattern
	Variable min_value

	Variable k
	String wname

	if (!ParamIsDefault(pattern) && strlen(pattern) > 0)
		if (strlen(wlist) > 0)
			wlist = AddListItem(WaveList(pattern, ";", ""), wlist)
		else
			wlist = WaveList(pattern, ";", "")
		endif
	endif
	wname = StringFromList(0, wlist)
	if (strlen(wname) == 0)
		return 1
	endif
	if (ParamIsDefault(min_value))
		min_value = -Inf
	endif
	Duplicate $wname, $target
	Wave twave = $target
	twave = 0
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		Wave w0 = $wname
		if (numtype(min_value) == 0)
			twave += (w0[p] > min_value) ? w0[p] : min_value 
		else
			twave += w0
		endif
	endfor
End

Function SimpleWaveOperation(w, [op_wave, op_var])
	Wave w, op_wave
	Variable op_var
End

Function divide_wave(w, [op_wave, op_var]) 
	Wave w, op_wave
	Variable op_var

	if (!ParamIsDefault(op_wave))
		w /= op_wave
	elseif (!ParamIsDefault(op_var))
		w /= op_var
	endif
End

Function subtract_wave(w, [op_wave, op_var]) 
	Wave w, op_wave
	Variable op_var

	if (!ParamIsDefault(op_wave))
		w -= op_wave
	elseif (!ParamIsDefault(op_var))
		w -= op_var
	endif
End

Function modify_waves(op_wave, output_folder, wave_op, [pattern, skip, op_wave_is_list])
	// Examples:
	//   modify_waves(buffer, "minus_buffer", subtract_wave, skip="Cycle")
	//   modify_waves(:::norm_95C_buffer, "minus_buffer", subtract_wave, skip="Cycle")
	Wave op_wave
	String output_folder, pattern, skip
	Funcref SimpleWaveOperation wave_op
	Variable op_wave_is_list

	Variable k
	String wlist, wname

	if (ParamIsDefault(pattern))
		pattern = ""
	endif
	if (ParamIsDefault(skip))
		skip = ""
	endif
	if (ParamIsDefault(op_wave_is_list))
		op_wave_is_list = 0
	endif

	skip = AddListItem(NameOfWave(op_wave), skip)
	NewDataFolder/O $output_folder
	wlist = WaveList("*", ";", "")
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		if (strlen(pattern) > 0 && stringmatch(wname, pattern) == 0)
			continue
		endif
		if (FindListItem(wname, skip) >= 0)
			continue
		endif
		Duplicate $wname :$(output_folder):$wname
		Wave wcopy = :$(output_folder):$wname
		if (op_wave_is_list)
			wave_op(wcopy, op_var=op_wave[%$wname])
		else
			wave_op(wcopy, op_wave=op_wave)
		endif
	endfor
End

Function divide_waves(divisor, [output_folder, pattern, skip, divisor_is_list])
	Wave divisor
	String output_folder, pattern, skip
	Variable divisor_is_list

	if (ParamIsDefault(output_folder))
		output_folder="div"
	endif
	if (ParamIsDefault(pattern))
		pattern = ""
	endif
	if (ParamIsDefault(skip))
		skip = ""
	endif
	if (ParamIsDefault(divisor_is_list))
		divisor_is_list = 0
	endif
	modify_waves(divisor, output_folder, divide_wave, pattern=pattern, skip=skip, op_wave_is_list=divisor_is_list)
End

Function divide_waves0(divisor, [output_folder, pattern, skip, divisor_is_list])
	Wave divisor
	String output_folder, pattern, skip
	Variable divisor_is_list

	Variable k
	String wlist, wname
//	String divisor_wname = NameOfWave(divisor)

	if (ParamIsDefault(output_folder))
		output_folder="div"
	endif
	if (ParamIsDefault(pattern))
		pattern = ""
	endif
	if (ParamIsDefault(skip))
		skip = ""
	endif
	if (ParamIsDefault(divisor_is_list))
		divisor_is_list = 0
	endif

	skip = AddListItem(NameOfWave(divisor), skip)
	NewDataFolder/O $output_folder
	wlist = WaveList("*", ";", "")
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		if (strlen(pattern) > 0 && stringmatch(wname, pattern) == 0)
			continue
		endif
		if (FindListItem(wname, skip) >= 0)
			continue
		endif

		Duplicate $wname :$(output_folder):$wname
		Wave wcopy = :$(output_folder):$wname
		if (divisor_is_list)
			wcopy /= divisor[%$wname]
		else
			wcopy /= divisor
		endif
	endfor
End

Function create_norm_wave(target, [p0, p1, pattern, sort])
	// Parameters:
	//   target:   name of the target wave
	//   p0, p1:   optional range to limit each wave to
	//   pattern:  matching pattern passed to WaveList(), defaults
	//             to "*"
	//   sort:     optional sort pattern
	//
	// When `sort` is not specified, waves are not sorted. Passing
	// an empty `sort` argument sorts using the default pattern
	// "%*[^0123456789]%d%*s".
	
	String target, pattern, sort
	Variable p0, p1

	Variable k, N
	String wname, wlist
	
	if (ParamIsDefault(pattern) || strlen(pattern) == 0)
		pattern = "*"
	endif
	if (ParamIsDefault(p0))
		p0 = 0
	endif
	if (ParamIsDefault(p1))
		p1 = NaN
	endif

	wlist = WaveList(pattern, ";", "")
	if (!ParamIsDefault(sort))
		if (strlen(sort) == 0)
			sort = "%*[^0123456789]%d%*s"
		endif
		wlist = sort_list_1(wlist, sort, ";")
	endif
	N = ItemsInList(wlist)
	if (N < 1)
		return 1
	endif
	Make/N=(N) $target
	Wave twave = $target

	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		if (numtype(p1) == 0)
			twave[k] = mean($wname, pnt2x($wname, p0), pnt2x($wname, p1))
		else
			twave[k] = mean($wname, pnt2x($wname, p0), Inf)
		endif
		SetDimLabel 0, k, $wname, twave
	endfor
End

Static Function make_scaled_duplicate(wname, max_value, suffix, [offset, target])
	// target:  target folder, defaults to current
	String wname, suffix, target
	Variable max_value, offset

	if (ParamIsDefault(offset))
		offset = 0
	endif
	if (ParamIsDefault(target))
		target = ""
	endif
	if (strlen(target) > 0)
		if (cmpstr(target[strlen(target) - 1], ":") != 0)
			target = target + ":"
		endif
		NewDataFolder/O $(target[0,strlen(target) - 2])
	endif

	String wname2
	wname2 = target + wname + suffix
//	if (WaveExists($wname2))
//		return 0
//	endif
//	Duplicate $wname $wname2
	if (WaveExists($wname2))
		Wave w = $wname2
		Wave w0 = $wname
		w = w0
	else
		Duplicate $wname $wname2
		Wave w = $wname2
	endif
	w *= max_value
	if (offset != 0)
		w += offset
	endif
	print "  WAVE created: ", GetDataFolder(1) + wname2
	return 1
End

Macro RescaleWaves(pattern, suffix, max_value, offset)
	String pattern = "*", suffix = ""
	Variable max_value = 1, offset = 0
	Prompt pattern, "Wave pattern:"
	Prompt suffix, "Target wave suffix:"
	Prompt max_value, "Target maximum value:"
	Prompt offset "Offset:"

//	printf "rescale_waves(\"%s\", \"%s\", %f, offset=%f)\r", pattern, suffix, max_value, offset
	rescale_waves(pattern, suffix, max_value, offset=offset)
End

Macro RescaleWavesFromGraph(pattern, suffix, max_value, offset)
	String pattern = "", suffix = ""
	Variable max_value = 1, offset = 0
	Prompt pattern, "Trace filter pattern:"
	Prompt suffix, "Target wave suffix:"
	Prompt max_value, "Target maximum value:"
	Prompt offset "Offset:"

//	printf "rescale_waves_from_graph(\"%s\", \"%s\", %f, offset=%f)\r", pattern, suffix, max_value, offset
	rescale_waves_from_graph(pattern, suffix, max_value, offset=offset)
End

Function rescale_waves(pattern, suffix, max_value, [offset, target])
	// Rescales waves using an affine transformation.
	//
	// All waves in the current data folder matching `pattern` are
	// rescaled from [0,1] to [offset, offset + max_value].
	//
	// Parameters:
	//   pattern:
	//   suffix:
	//   max_value:  the target maximum value
	//   offset:     defaults to 0
	//   target:     target folder, defaults to current

	String pattern, suffix, target
	Variable max_value, offset

	if (ParamIsDefault(target))
		target = ""
	endif
	if (strlen(suffix) == 0 && strlen(target) == 0)
		Abort "no suffix and no target directory specified."
	endif
	if (ParamIsDefault(offset))
		offset = 0
	endif
	if (max_value == 1 && offset == 0)
		Abort "max_value == 1 && offset == 0, nothing to do."
	endif

	Variable k
	String wname, wlist

	wlist = WaveList(pattern, ";", "")
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		make_scaled_duplicate(wname, max_value, suffix, offset=offset, target=target)
	endfor
End

Function rescale_waves_from_graph(pattern, suffix, max_value, [offset])
	String pattern, suffix
	Variable max_value, offset

	if (strlen(suffix) == 0)
		Abort "no suffix specified."
	endif
	if (ParamIsDefault(offset))
		offset = 0
	endif
	if (max_value == 1 && offset == 0)
		Abort "max_value == 1 && offset == 0, nothing to do."
	endif

	Variable k
	String saved_folder, traces, trace, wname

	traces = TraceNameList("", ";", 1)
	saved_folder = GetDataFolder(1)
	for (k = 0;; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		Wave w = TraceNameToWaveRef("", trace)
		wname = NameOfWave(w)
		if (strlen(pattern) > 0 && !stringmatch(wname, pattern))
			continue
		endif
		SetDataFolder $GetWavesDataFolder(w, 1)
		make_scaled_duplicate(wname, max_value, suffix, offset=offset)
	endfor
	SetDataFolder $saved_folder
End

Function make_bar_waves(wx, wy, [err, width])
	Wave wx, wy, err
	Variable width

	Variable n
	if (ParamIsDefault(err))
		Wave err = $""
	endif
	if (ParamIsDefault(width))
		width = 0.8
	endif

	n = numpnts(wy)
	Differentiate/METH=1/EP=1 wx /D=$(NameOfWave(wx) + "_bar")
	wave wx2 = $(NameOfWave(wx) + "_bar")
	WaveStats/Q wx2
	if (n == 1)
		V_min = 1
	endif
	Redimension/N=(3*n + 2) wx2
	wx2 = wx[floor((p-1)/3)]
	wx2[0] -= (1 - width / 2) * V_min
	wx2[3*n + 1] += V_min
	wx2[1, ;3] -= width / 2 * V_min
	wx2[3, ;3] += width / 2 * V_min

	Make/O/N=(3*n + 2) $(NameOfWave(wy) + "_bar") = (mod(p+2,3) != 2) ? wy[floor((p-1)/3)] : NaN
	if (WaveExists(err))
		Make/O/N=(3*n + 2) $(NameOfWave(err) + "_bar") = (mod(p+2,3) == 1) ? err[floor((p-1)/3)] : NaN
	endif
End

Function/S quote_data_folder(name)
	String name

	String ret, part
	Variable k, n

	if (cmpstr(name[0,3], "root") != 0 && cmpstr(name[0], ":") != 0)
		name = ":" + name
	endif

	// Count the items as there may be empty ones.
	n = ItemsInList(name, ":")
	ret = PossiblyQuoteName(StringFromList(0, name, ":"))
	for (k = 1; k < n; k += 1)
		part = StringFromList(k, name, ":")
		ret = ret + ":" + PossiblyQuoteName(part)
	endfor
	return ret + ":"
End

Function concatenate_2d(format_str, target)
	String format_str, target

	String wlist
	Variable m, n, k
	wlist = WaveList(ReplaceString("%d", format_str, "*"), ";", "")
	wlist = sort_list_1(wlist, format_str, ";") + ";"
	n = numpnts($StringFromList(0, wlist))
	m = ItemsInList(wlist)
	Concatenate wlist, $target
	Redimension/N=(n,m) $target
	for (k = 0; k < m; k += 1)
		SetDimLabel 1, k, $StringFromList(k, wlist), $target
	endfor
End

Function combine_waves(path1, path2, output, [pattern])
	// Combines waves from two paths.
	//
	// The resulting waves in folder `output` are created using
	//   Concatenate/NP `path1:wave_N` `path2:wave_N` `output:wave_N`.
	String path1, path2, output, pattern

	Variable k
	String data_folder, wlist1, wlist2, wname

	if (ParamIsDefault(pattern))
		pattern = "*"
	endif

	data_folder = GetDataFolder(1)
	SetDataFolder $path1
	wlist1 = WaveList(pattern, ";", "")
	SetDataFolder $data_folder
	SetDataFolder $path2
	wlist2 = WaveList(pattern, ";", "")
	SetDataFolder $data_folder
	NewDataFolder/O $output

	path1 = quote_data_folder(path1)
	path2 = quote_data_folder(path2)
	output = quote_data_folder(output)

	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist1)
		if (strlen(wname) == 0)
			break
		endif
		if (FindListItem(wname, wlist2) < 0)
			continue
		endif
		Concatenate/NP {$(path1 + wname), $(path2 + wname)}, $(output + wname)
	endfor
End

Function interleave_waves(pattern, output_name)
	// Interleaves waves.
	//
	// The resulting wave will look like
	//   {wave0[0], wave1[0], ..., waveN[0], wave0[1], wave1[1], ..., waveN[m]}
	// where 'm' is the length of the shortest wave.
	//
	// Example: interleave_waves("step*", "all_steps")
	String pattern, output_name

	Variable k, nw, n0
	String folder, folder0, wlist, wname

	wlist = WaveList(pattern, ";", "")
	for (nw = 0, n0 = Inf, k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		nw += 1
		n0 = min(n0, numpnts($wname))
	endfor
	Make/N=(nw * n0) $output_name
	Wave target = $output_name
//	target = NaN
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		Wave src = $wname
//		print src
		target[k,*;nw] = src[floor(p / nw)]
	endfor
//	print target
End

Function MergeTraceData()
	String color = "yes", lSize = "yes", lStyle = "yes"
	String mode = "yes", marker = "yes", msize = "yes"

	Prompt color, "color", popup "yes;no"
	Prompt lSize, "line size", popup "yes;no"
	Prompt lStyle, "line style", popup "yes;no"
	Prompt mode, "mode", popup "yes;no"
	Prompt marker, "marker type", popup "yes;no"
	Prompt msize, "marker size", popup "yes;no"
	DoPrompt "Merge trace data based on style properties...", color, lSize, lStyle, mode, marker, msize

	String style_list = "";
	if (cmpstr(color, "yes") == 0)
		style_list = AddListItem("rgb", style_list)
	endif
	if (cmpstr(lSize, "yes") == 0)
		style_list = AddListItem("lSize", style_list)
	endif
	if (cmpstr(lStyle, "yes") == 0)
		style_list = AddListItem("lStyle", style_list)
	endif
	if (cmpstr(mode, "yes") == 0)
		style_list = AddListItem("mode", style_list)
	endif
	if (cmpstr(marker, "yes") == 0)
		style_list = AddListItem("marker", style_list)
	endif
	if (cmpstr(msize, "yes") == 0)
		style_list = AddListItem("msize", style_list)
	endif
	merge_trace_data(style_list=style_list)
End

Function/S get_error_bars()
	Variable k, len, pos, k2
	String win, line, trace, spec, err, err2, err_list

	win = WinRecreation("", 0)
	err_list = ""
	for (k = 0;; k += 1)
		line = StringFromList(k, win, "\r\n")
		if (strlen(line) < 1)
			break
		endif
		pos = strsearch(line, "ErrorBars", 0)
		if (pos < 0)
			continue
		endif
		trace = StringFromList(1, line, " ")
		spec = StringFromList(2, line, " ")
		if (cmpstr(StringFromList(0, spec, "="), "Y,wave") == 0)
			spec = StringFromList(1, spec, "=")
			len = strlen(spec)
			spec = spec[1,len-2]
			err = parse_wave_list(spec, k2, start=0, sep=",")
			err2 = spec[k2+1,len-3]
			if (cmpstr(err, err2) == 0)
				err_list = AddListItem(trace + "=" + err, err_list)
			else
				Abort "This is not implemented."
			endif
		endif
	endfor
	return err_list
End

Function get_trace_style_groups(output_wname, [traces, style_list, values_from])
	String output_wname, traces, style_list, values_from

	String trace, style
	Variable n_styles, n_traces, k, ks, value_idx

	if (ParamIsDefault(traces) || strlen(traces) == 0)
		traces = TraceNameList("", ";", 1)
	endif
	if (ParamIsDefault(style_list))
		style_list = ""
	endif

	value_idx = -1
	if (ParamIsDefault(values_from))
		values_from = ""
	elseif (strlen(values_from) > 0)
		if (strlen(style_list) == 0)
			style_list = values_from
			value_idx = 0
		else
			value_idx = FindListItem(values_from, style_list)
		endif
	endif
	if (strlen(style_list) == 0)
		style_list = "rgb;lSize;lStyle;mode;marker;msize"
	endif

	n_traces = ItemsInList(traces)
	Make/T/N=(n_traces) tmp_trace_styles
	Make/O/N=(n_traces) $output_wname
	Wave ow = $output_wname

	n_styles = 0
	for (k = 0; k < n_traces; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		style = trace_get_style(trace, style_list, sep=";", values_only=1)

		for (ks = 0; ks < n_styles; ks += 1)
			if (cmpstr(tmp_trace_styles[ks], style) == 0)
				break
			endif
		endfor
		if (ks == n_styles)
			n_styles += 1
			tmp_trace_styles[ks] = style
		endif
		if (strlen(values_from) > 0)
			if (value_idx < 0)
				ow[k] = str2num(trace_get_style(trace, values_from, sep=";", values_only=1))
			else
				ow[k] = str2num(StringFromList(value_idx, style))
			endif
		else
			ow[k] = ks
		endif
	endfor
	KillWaves tmp_trace_styles
End

Function merge_trace_data([style_list])
	// Merges data from traces with the same style.
	//
	// Parameters:
	//   style_list:  ";"-separated list of styles to consider,
	//                defaults to "rgb;lSize;lStyle;mode;marker;msize"
	String style_list;

	Variable k, ks, k0, k1, k2, n_traces, n_styles, n_style_items, x0, x1, y0, y1
	String trace, traces, style, name, prefix, err_traces, err

	if (ParamIsDefault(style_list) || strlen(style_list) == 0)
		style_list = "rgb;lSize;lStyle;mode;marker;msize"
	endif

	traces = TraceNameList("", ";", 1)
	n_traces = ItemsInList(traces)
	Make/T/N=(n_traces) tmp_trace_styles
	Make/N=(n_traces) tmp_trace_style_idx, tmp_trace_idx, tmp_trace_start, tmp_trace_len

	n_styles = 0
	for (k = 0; k < n_traces; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		style = trace_get_style(trace, style_list, sep=";", values_only=1)
		trace_get_ranges(trace, x0, x1, y0, y1)

		for (ks = 0; ks < n_styles; ks += 1)
			if (cmpstr(tmp_trace_styles[ks], style) == 0)
				break
			endif
		endfor
		if (ks == n_styles)
			n_styles += 1
			tmp_trace_styles[ks] = style
			tmp_trace_style_idx[ks] = k
		endif
		tmp_trace_idx[k] = ks
		tmp_trace_start[k] = tmp_trace_len[ks]
		tmp_trace_len[ks] += x1 - x0 + 1
	endfor

	err_traces = get_error_bars()
	for (ks = 0; ks < n_styles; ks += 1)
		prefix = "wave" + num2str(ks)
		Make/N=(tmp_trace_len[ks]) $(prefix + "_x"), $(prefix + "_y"), $(prefix + "_err") = NaN
	endfor
	for (k = 0; k < n_traces; k += 1)
		trace = StringFromList(k, traces)
		prefix = "wave" + num2str(tmp_trace_idx[k])

		trace_get_ranges(trace, x0, x1, y0, y1)
		k0 = tmp_trace_start[k]
		k1 = k0 + y1 - y0

		Wave target_x = $(prefix + "_x")
		Wave target_y = $(prefix + "_y")
		Wave source_x = XWaveRefFromTrace("", trace)
		Wave source_y = TraceNameToWaveRef("", trace)

		target_x[k0,k1] = source_x[p+x0-k0]
		target_y[k0,k1] = source_y[p+y0-k0]

		err = StringByKey(trace, err_traces, "=")
		if (strlen(err) > 0)
			Wave target_err = $(prefix + "_err")

			// The path in `err` is relative to root, but
			// the current data folder may be different.
			err = "root" + err
			k2 = strsearch(err, "[", 0)
			if (k2 < 0)
				Wave source_err = $err
				target_err[k0,k1] = source_err
			else
				Wave source_err = $(err[0,k2-1])
				parse_range(source_err, err[k2,strlen(err)-1], y0, y1)
				target_err[k0,k1] = source_err[y0]
			endif
		endif
	endfor

	name = WinName(0, 1)
	Display as name + " (2)"
	DoWindow/C $(name + "_2")
	for (ks = 0; ks < n_styles; ks += 1)
		prefix = "wave" + num2str(ks)
		trace = prefix + "_y"
		err = prefix + "_err"
		AppendToGraph $trace vs $(prefix + "_x")
		ErrorBars $trace Y,wave=($err,$err)
		k = tmp_trace_style_idx[ks]
		trace_copy_style(name, StringFromList(k, traces), trace)
	endfor
	KillWaves tmp_trace_styles, tmp_trace_style_idx, tmp_trace_idx, tmp_trace_start, tmp_trace_len
End

Function/S filter_traces_by_style(traces, style_list)
	// Parameters:
	//   traces:      ";"-separated list of trace styles
	//   style_list:  ","-separated list of attr=value style properties,
	//                eg. "lsize=1,mSize=2"
	String traces, style_list

	Variable k
	String trace, matching_traces

	matching_traces = ""
	for (k = 0;; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		if (trace_matches_style(trace, style_list))
			matching_traces = AddListItem(trace, matching_traces)
		endif
	endfor
	return matching_traces
End

Function flat_baseline([output_folder])
	String output_folder

	Variable k, v0, v1
	String wlist, wname

	if (ParamIsDefault(output_folder))
		output_folder="baseline"
	endif

	NewDataFolder/O $output_folder
	wlist = WaveList("*", ";", "")
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		WaveStats/Q/M=1 $wname
		Duplicate $wname :$(output_folder):$wname
		Wave wcopy = :$(output_folder):$wname
		wcopy -= V_min
	endfor
End

Function norm_waves([output_folder, pattern, total, index])
	String output_folder, pattern
	Variable index, total

	Variable k, v0, v1
	String wlist, wname

	if (ParamIsDefault(output_folder))
		output_folder = "norm"
	endif
	if (ParamIsDefault(pattern))
		pattern = "*"
	endif
	if (ParamIsDefault(total) || numtype(total) != 0)
		total = 0
	endif
	if (ParamIsDefault(index) || total)
		index = NaN
	endif

	NewDataFolder/O $output_folder
	wlist = WaveList(pattern, ";", "")
	for (k = 0;; k += 1)
		wname = StringFromList(k, wlist)
		if (strlen(wname) == 0)
			break
		endif
		Duplicate $wname :$(output_folder):$wname
		Wave wcopy = :$(output_folder):$wname
		if (total)
			v0 = sum(wcopy)
			wcopy /= v0
		elseif (numtype(index) == 0)
			if (index < 0)
				v0 = wcopy[DimSize(wcopy, 0) - index]
			else
				v0 = wcopy[index]
			endif
			wcopy /= v0
		else
			v0 = wcopy[0]
			v1 = wcopy[DimSize(wcopy, 0) - 1]
			wcopy -= min(v0, v1)
			wcopy /= abs(v0 - v1)
		endif
	endfor
End

Macro GetValuesAtX(x0, output_wname)
	Variable x0 = 0
	String output_wname = "output_wave"
	Prompt x0, "x coordinate"
	Prompt output_wname, "Name of the output wave"

	get_values_at_x(x0, output_wname)
End

Function get_values_at_x(x0, output_wname, [pattern, style])
	Variable x0
	String output_wname, pattern, style

	Variable k, n, p0, x1, x2, y1, y2
	String trace, traces, all_traces

	if (ParamIsDefault(pattern))
		pattern = ""
	endif
	if (ParamIsDefault(style))
		style = ""
	endif

	traces = TraceNameList("", ";", 1)
	if (strlen(style) > 0)
		traces = filter_traces_by_style(traces, style)
	endif
	if (strlen(pattern) > 0)
		all_traces = traces
		traces = ""
		for (k = 0, n = 0;; k += 1)
			trace = StringFromList(k, all_traces)
			if (strlen(trace) < 1)
				break
			endif
			if (stringmatch(trace, pattern))
				n += 1
				traces = AddListItem(trace, traces)
			endif
		endfor
	else
		n = ItemsInList(traces)
	endif
	if (n < 1)
		return NaN
	endif

	Make/N=(n) $output_wname
	Wave ow = $output_wname

	SetDimLabel 0, -1, $WinName(0, 1), ow
	for (k = 0; k < n; k += 1)
		trace = StringFromList(k, traces)

		Wave xw = XWaveRefFromTrace("", trace)
		Wave yw = TraceNameToWaveRef("", trace)
		if (WaveExists(xw))
			trace_get_ranges(trace, x1, x2, y1, y2)
			p0 = find_x2pnt(xw, x0, p0=x1, p1=x2)
		else
			p0 = x2pnt(yw, x0)
		endif
		ow[k] = yw[y1+p0]
		SetDimLabel 0, k, $trace, ow
	endfor		
End

Function find_x2pnt(xwave, x0, [p0, p1])
	// Finds the index in `xwave` where its value is closest to `x0`.
	Wave xwave
	Variable x0, p0, p1

	if (ParamIsDefault(p0))
		p0 = 0
	endif
	if (ParamIsDefault(p1))
		p1 = inf
	endif
	Duplicate/O/R=[p0,p1] xwave, tmp_find_x2pnt
	tmp_find_x2pnt = abs(xwave - x0)
	WaveStats/Q tmp_find_x2pnt
	KillWaves tmp_find_x2pnt
	return V_minloc
end

Function melting_curve_baselines(ywave, xwave, x0, x1, [xmin, xmax, trace])
	Wave ywave, xwave
	Variable x0, x1, xmin, xmax
	String trace

	String folder
	Variable p0, p1, pmin, pmax, p_tmp, len, tmp_xwave, is_reverse

	if (!ParamIsDefault(trace))
		Wave ywave = TraceNameToWaveRef("", trace)
		if (!WaveExists(ywave))
			String error_msg
			sprintf error_msg, "Trace \"%s\" is not on the graph.", trace
			Abort error_msg
		endif
		Wave xwave = XWaveRefFromTrace("", trace)
	endif
	len = numpnts(ywave)
	tmp_xwave = !WaveExists(xwave)
	if (tmp_xwave)
		is_reverse = DimDelta(ywave, 0) < 0
	else
		is_reverse = (xwave[0] > xwave[len - 1])
	endif
	if (ParamIsDefault(xmin))
		if (is_reverse)
			pmin = len - 1
		else
			pmin = 0
		endif
	else
		pmin = find_x2pnt(xwave, xmin)
	endif
	if (ParamIsDefault(xmax))
		if (is_reverse)
			pmax = 0
		else
			pmax = len - 1
		endif
	else
		pmax = find_x2pnt(xwave, xmax)
	endif
	if (pmax < pmin)
		p_tmp = pmin
		pmin = pmax
		pmax = p_tmp
	endif

	folder = GetDataFolder(1)
	SetDataFolder GetWavesDataFolder(ywave, 1)
	String yname = NameOfWave(ywave)
	String yname2 = yname + "_melt"
	String baseline1 = yname + "_base1"
	String baseline2 = yname + "_base2"

	Duplicate/O ywave, $yname2, $baseline1, $baseline2
	Wave ywave2 = $yname2

	If (tmp_xwave)
		// Use wave scaling
		p0 = x2pnt(ywave, x0)
		p1 = x2pnt(ywave, x1)
		// Make a temporary xwave
		Duplicate ywave, $(yname + "_x")
		Wave xwave = $(yname + "_x")
		xwave = pnt2x(ywave, p)
	else
		p0 = find_x2pnt(xwave, x0)
		p1 = find_x2pnt(xwave, x1)
	endif

	if (p1 < p0)
		p_tmp = p0
		p0 = p1
		p1 = p_tmp
	endif

	CurveFit/Q/X line ywave[pmin,p0] /X=xwave
	Wave bl1 = $baseline1
	bl1 = K0 + K1 * xwave
	CurveFit/Q/X line ywave[p1,pmax] /X=xwave
	Wave bl2 = $baseline2
	bl2 = K0 + K1 * xwave

	ywave2 = (ywave - bl1) / (bl2 - bl1)
	if (is_reverse)
		ywave2 = 1 - ywave2
	endif
	if (tmp_xwave)
		KillWaves xwave
		Display $baseline1[pmin,pmax],$baseline2[pmin,pmax],ywave[pmin,pmax] as "Melting curve: " + yname
	else
		Display $baseline1[pmin,pmax],$baseline2[pmin,pmax],ywave[pmin,pmax] vs xwave[pmin,pmax] as "Melting curve: " + yname
	endif
	ModifyGraph lsize($yname)=1.5,rgb($yname)=(0,0,0)
	ModifyGraph rgb($baseline1)=(43520,43520,43520),rgb($baseline2)=(43520,43520,43520)
	ModifyGraph width=150,height=120
	ModifyGraph width=0,height=0

	if (tmp_xwave)
		Display ywave2[pmin,pmax]
	else
		Display ywave2[pmin,pmax] vs xwave[pmin,pmax]
	endif
	ModifyGraph zero(left)=1
	SetDataFolder folder
End

Function set_trace_styles(property, value, pattern, [traces])
	String property, value, pattern, traces
	String trace, cmd
	Variable k

	if (ParamIsDefault(traces))
		traces = ""
	endif
	if (strlen(traces) == 0)
		traces = TraceNameList("", ";", 1)
	endif
	for (k = 0;; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		if (strlen(pattern) > 0 && !stringmatch(trace, pattern))
			continue
		endif
		sprintf cmd, "ModifyGraph %s(%s)=%s\r\n", property, trace, value
		Execute/Q cmd
	endfor
End

Function parse_range(w, range, a, b)
	Wave w
	String range
	Variable &a, &b
	if (cmpstr(range, "[*]") == 0)
		a = 0
		b = numpnts(w) - 1
	else
		sscanf range, "[%d,%d]", a, b
	endif
End

Function/T parse_wave_list(wave_list, end_idx, [start, sep])
	String wave_list, sep
	Variable &end_idx, start

	Variable k, n, len
	String s

	if (ParamIsDefault(start))
		start = 0
	endif
	if (ParamIsDefault(sep))
		sep = ";"
	endif
	if (cmpstr(sep, ","))
		k = strsearch(wave_list, sep, start)
		end_idx = k
		if (k < 0)
			return wave_list
		endif
		return wave_list[start,k-1]
	endif

	// Assume that brackets are correctly nested.
	len = strlen(wave_list)
	for (n = 0, k = start; k < len; k += 1)
		s = wave_list[k]
		if (strsearch("([", s, 0) >= 0)
			n += 1
		elseif (strsearch(")]", s, 0) >= 0)
			n -= 1
		elseif (cmpstr(s, sep) == 0 && n == 0)
			end_idx = k
			return wave_list[start,k-1]
		endif
	endfor
End

Function trace_get_ranges(trace, x0, x1, y0, y1)
	String trace
	Variable &x0, &x1, &y0, &y1

	String tinfo, message
	String xrange, yrange

	tinfo = TraceInfo("", trace, 0)
	xrange = StringByKey("XRANGE", tinfo)
	if (stringmatch(xrange, "*:*"))
		String xwave
		xwave = StringByKey("XWAVE", tinfo)
		sprintf message, "Cannot operate on traces with non-continuous XRANGE: %s%s", xwave, xrange
		Abort message
	endif

	Wave xw = XWaveRefFromTrace("", trace)
	parse_range(xw, xrange, x0, x1)
	yrange = StringByKey("YRANGE", tinfo)
	if (stringmatch(yrange, "*:*"))
		sprintf message, "Cannot operate on traces with non-continuous YRANGE: %s%s", trace, yrange
		Abort message
	endif
	Wave yw = TraceNameToWaveRef("", trace)
	parse_range(yw, yrange, y0, y1)
End

Function/S trace_get_style(trace, attr_list, [sep, values_only])
	// Parameters:
	//   trace:        trace name
	//   attr_list:    ";"-separated list of attributes, eg. "rgb;lSize"
	//   sep:          separator in the output list, defaults to ","
	//   values_only:  whether to return only the values of the given
	//                 style attributes instead of attr=value pairs
	String trace, attr_list, sep
	Variable values_only
	String trace_info, attr, value, style, item
	Variable k

	if (ParamIsDefault(sep))
		sep = ","
	endif
	if (ParamIsDefault(values_only))
		values_only = 0
	endif

	style = ""
	trace_info = TraceInfo("", trace, 0)
//	sep = "(" + trace + ")="
	for (k = 0;; k += 1)
		attr = StringFromList(k, attr_list)
		if (strlen(attr) == 0)
			break
		endif
		value = StringByKey(attr + "(x)", trace_info, "=", ";")
		// Do not use AddListItem as that adds a separator to the end.
//		style = AddListItem(attr + "=" + value, style, ",")
		if (values_only)
			item = value
		else
			item = attr + "=" + value
		endif
		if (strlen(style) == 0)
			style = item
		else
			style = style + sep + item
		endif
	endfor
	return style
End

Function trace_matches_style(trace, style_list, [sep])
	// Checks whether a trace matches a given style
	//
	// Parameters:
	//   trace:       trace name
	//   style_list:  list of attr=value pairs of style properties,
	//                eg. "lsize=1,mSize=2"
	//   sep:         separator in `style_list`, defaults to ","
	String trace, style_list, sep

	Variable k
	String trace_info, style, attr, value

	if (ParamIsDefault(sep) || strlen(sep) == 0)
		sep = ","
	endif

	trace_info = TraceInfo("", trace, 0)
	for (k = 0;; k += 1)
		style = StringFromList(k, style_list, sep)
		attr = StringFromList(0, style, "=")
		if (strlen(attr) == 0)
			break
		endif
		value = StringByKey(attr + "(x)", trace_info, "=", ";")
		if (cmpstr(value, StringFromList(1, style, "=")) != 0)
			return 0
		endif
	endfor
	return 1
End

Function trace_copy_style(src_window, src_trace, dest_trace)
	String src_window, src_trace, dest_trace
	String tinfo, style, item, key, cmd
	Variable k

	tinfo = TraceInfo(src_window, src_trace, 0)

	k = strsearch(tinfo, "RECREATION", 0)
	style = tinfo[k+11,strlen(tinfo)]
	key = "(" + dest_trace + ")"
	for (k = 0;; k += 1)
		item = StringFromList(k, style)
		if (strlen(item) == 0)
			break
		endif
		cmd = "ModifyGraph " + ReplaceString("(x)", item, key)
		Execute/Q cmd
	endfor
End

Function trace_index(trace)
	String trace
	String s
	Variable index

	s = StringFromList(1, trace, "#")
	if (strlen(s) == 0)
		index = 0
	else
		sscanf s, "%d", index
	endif
	return index
End

Function/S get_common_data_folder(traces)
	String traces
	String trace, folder, f0, p0, p1
	Variable k, kp, len

	folder = ""
	for (k = 0;; k += 1)
		trace = StringFromList(k, traces)
		if (strlen(trace) == 0)
			break
		endif
		Wave w = TraceNameToWaveRef("", trace)
		f0 = GetWavesDataFolder(w, 1)
		if (strlen(folder) == 0)
			folder = f0
			continue
		endif
		if (cmpstr(folder, f0) != 0)
			for (len = 0, kp = 0;; kp += 1)
				p0 = StringFromList(kp, folder, ":")
				p1 = StringFromList(kp, f0, ":")
				if (cmpstr(p0, p1) != 0)
					break
				endif
				len += strlen(p0) + 1
			endfor
			if (len > 0)
				folder = folder[0,len-1]
			endif
		endif
	endfor
	return folder
End

Function/S unescape_string(text)
	// FIXME: Is there really not a predefined function for this?
	String text
	text = ReplaceString("\\t", text, "\t")
	text = ReplaceString("\\r", text, "\r")
	text = ReplaceString("\\n", text, "\n")
	text = ReplaceString("\\\\", text, "\\")
	return text
End

Function/S get_legend_text(legend_name)
	String &legend_name
	String alist, ainfo, aname, text
	Variable k

	text = ""
	alist = AnnotationList("")
	for (k = 0;; k += 1)
		aname = StringFromList(k, alist)
		if (strlen(aname) == 0)
			break
		endif
		ainfo = AnnotationInfo("", aname)
		if (cmpstr(StringByKey("TYPE", ainfo), "Legend") == 0)
			legend_name = aname
			text = unescape_string(StringByKey("TEXT", ainfo))
			break
		endif
	endfor
	return text
End
